<?php
require 'db.php';

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

$data = json_decode(file_get_contents("php://input"), true);

if (!$data) {
    echo json_encode(array("message" => "Invalid JSON input"));
    exit;
}

if (isset($data['session_id']) && isset($data['participants'])) {
    $session_id = $data['session_id'];
    $participants = $data['participants'];
    $images = isset($data['images']) ? $data['images'] : [];

    // Initialize counters
    $below_16_years = 0;
    $between_16_to_60_years = 0;
    $above_60_years = 0;
    $male = 0;
    $female = 0;
    $other = 0;

    // Verify if session_id exists and capture yoga_instructor_id
    $verify_stmt = $conn->prepare("SELECT yoga_instructor_id FROM yog_sessions WHERE id = ?");
    $verify_stmt->bind_param("i", $session_id);
    $verify_stmt->execute();
    $verify_stmt->store_result();
    $verify_stmt->bind_result($yoga_instructor_id);

    if ($verify_stmt->num_rows > 0) {
        $verify_stmt->fetch();
        $conn->begin_transaction();
        $error = false;
        $errorMessage = '';

        foreach ($participants as $participant) {
            $stmt = $conn->prepare("INSERT INTO participants (session_id, yog_instructor_id, name, mobile, age, gender) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("iissss", $session_id, $yoga_instructor_id, $participant['name'], $participant['mobile'], $participant['age'], $participant['gender']);
            
            if (!$stmt->execute()) {
                $error = true;
                $errorMessage = $stmt->error;
                error_log("Failed to execute statement: " . $stmt->error);
                break;
            }

            // Update counters based on participant details
            $age = intval($participant['age']);
            $gender = strtolower($participant['gender']);
            if ($age < 16) {
                $below_16_years++;
            } elseif ($age <= 60) {
                $between_16_to_60_years++;
            } else {
                $above_60_years++;
            }

            if ($gender == 'male') {
                $male++;
            } elseif ($gender == 'female') {
                $female++;
            } else {
                $other++;
            }

            $stmt->close();
        }

        if (!$error) {
            // Update session images and participant details
            $images_json = json_encode($images);
            $participant_number = count($participants);
            $update_stmt = $conn->prepare("UPDATE yog_sessions SET 
                Session_Image = ?, 
                participant_number = ?, 
                Below_16_years = ?, 
                Between_16_to_60_years = ?, 
                Above_60_years = ?, 
                Male = ?, 
                Female = ?, 
                other = ? 
                WHERE id = ?");
            $update_stmt->bind_param("siiiiissi", $images_json, $participant_number, $below_16_years, $between_16_to_60_years, $above_60_years, $male, $female, $other, $session_id);

            if (!$update_stmt->execute()) {
                $error = true;
                $errorMessage = $update_stmt->error;
                error_log("Failed to update session images and details: " . $update_stmt->error);
            }
            $update_stmt->close();
        }

        if ($error) {
            $conn->rollback();
            echo json_encode(array("message" => "Failed to add participants: " . $errorMessage));
        } else {
            $conn->commit();
            echo json_encode(array("message" => "Participants and session details added successfully"));
        }
    } else {
        echo json_encode(array("message" => "Invalid session_id"));
    }

    $verify_stmt->close();
} else {
    echo json_encode(array("message" => "Invalid input"));
}

$conn->close();
?>
